<?php
/* --------------------------------------------------------------
  Authentication.inc.php 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/

namespace StyleEdit;

/**
 * Class Authentication
 * 
 * @package StyleEdit
 */
class Authentication
{
	/**
	 * Check if the user is authenticated for using StyleEdit.
	 * 
	 * @return bool
	 */
	public static function isAuthenticated()
	{
	    $authenticated = true;
		if(!defined('UNIT_TEST_RUNNING'))
		{
		    $authenticated = false;
            if(isset($_COOKIE[\StyleEditConfig::AUTH_SESSION_KEY]) &&
                $_COOKIE[\StyleEditConfig::AUTH_SESSION_KEY] === self::getSecureToken())
            {
                $authenticated = true;
            }
		}
		
		return $authenticated;
	}
	
	
	/**
	 * Stop script, send 403-Header and display message on screen.
	 */
	public static function showForbiddenPage()
	{
		header('HTTP/1.0 403 Forbidden');
		die('<h1>403 Forbidden</h1>
			 <a href="..">Access denied, please authenticate.</a>');
	}
	
	
	/**
	 * Set authentication cookie.
	 *
	 * IMPORTANT: Only call this method if the user is already authorized to use the application
	 */
	public static function setAuthenticationToValid()
	{
	    setcookie(
	        \StyleEditConfig::AUTH_SESSION_KEY,
            self::getSecureToken(),
            0,
            '/'
        );
	}
	
	
	/**
	 * Deletes authentication cookie
	 */
	public static function setAuthenticationToInvalid()
	{
        setcookie(
            \StyleEditConfig::AUTH_SESSION_KEY,
            self::getSecureToken(),
            time() - 3600,
            '/'
        );
	}
    
    
    /**
     * Returns the shop’s secure token
     *
     * @return string
     */
	protected static function getSecureToken()
    {
        $tokenFiles = glob(\StyleEditConfig::getMediaPath() . 'secure_token_*');
        if(!is_array($tokenFiles) || count($tokenFiles) !== 1)
        {
            throw new \RuntimeException('Error locating secure_token file');
        }
        $token = '';
        foreach($tokenFiles as $tokenFile)
        {
            $token = (string)str_replace('secure_token_', '', basename($tokenFile));
        }
        return $token;
    }
}
